const http = require('http');
const WebSocket = require('ws');
const pg = require('pg');
const dbconfig = require('../db');
const pool = new pg.Pool(dbconfig);
const server = http.createServer();

// WebSocket Namespace for `market_price_update_api`
const marketPriceNamespace = new WebSocket.Server({ noServer: true });

marketPriceNamespace.on('connection', (ws) => {
  console.log('Client connected to market_price_update_api');
  // Add specific logic for this namespace if needed
});

// WebSocket Namespace for `marketlist_api`
const marketlistNamespace = new WebSocket.Server({ noServer: true });

marketlistNamespace.on('connection', (ws) => {
  console.log('Client connected to marketlist_api');

  const sendMarketData = async () => {
    try {
      const result = await pool.query('SELECT * FROM tbl_market_live_rate WHERE status = 1 order by tbl_market_live_rate_id DESC');
      const marketData = result.rows;

      // Send all data as a single JSON object instead of separate messages
      ws.send(JSON.stringify({ event: 'market_live_rate_data', data: marketData }));

    } catch (err) {
      console.error('Error querying tbl_market_live_rate:', err);
      ws.send(JSON.stringify({ event: 'market_data', error: 'Failed to fetch market data' }));
    }
  };

  // Send market data immediately upon connection
  sendMarketData();

  // Set an interval to send market data every 5 seconds
  const intervalId = setInterval(sendMarketData, 500);

  // Clear interval when client disconnects
  ws.on('close', () => {
    clearInterval(intervalId);
  });
});

// Handle HTTP upgrade to WebSocket and route to correct namespace
server.on('upgrade', (request, socket, head) => {
  const pathname = request.url;

  if (pathname === '/market_price_update_api') {
    marketPriceNamespace.handleUpgrade(request, socket, head, (ws) => {
      marketPriceNamespace.emit('connection', ws, request);
    });
  } else if (pathname === '/marketlist_api') {
    marketlistNamespace.handleUpgrade(request, socket, head, (ws) => {
      marketlistNamespace.emit('connection', ws, request);
    });
  } else {
    socket.destroy();
  }
});

// Start the HTTP server
server.listen(8158, function listening() {
  console.log('Server is listening on port 8136');
});
