const http = require('http');
const WebSocket = require('ws');
const pg = require('pg');
const url = require('url'); // <-- required to parse query params
const dbconfig = require('../db');
const pool = new pg.Pool(dbconfig);
const server = http.createServer();

// WebSocket Namespace for `market_price_update_api`
const marketPriceNamespace = new WebSocket.Server({ noServer: true });

marketPriceNamespace.on('connection', (ws) => {
  console.log('Client connected to market_price_update_api');
  // You can add price update logic here
});

// WebSocket Namespace for `marketlist_api`
const marketlistNamespace = new WebSocket.Server({ noServer: true });

marketlistNamespace.on('connection', (ws, request) => {
  console.log('Client connected to marketlist_api');

  const parsedUrl = url.parse(request.url, true);
  const tbl_user_id = parsedUrl.query.tbl_user_id;

  if (!tbl_user_id) {
    ws.send(JSON.stringify({ event: 'error', error: 'Missing tbl_user_id' }));
    ws.close();
    return;
  }

  const sendMarketData = async () => {
    try {
      const result = await pool.query(
        `SELECT tbl_market_live_rate.*  
         FROM tbl_watch_list 
         INNER JOIN tbl_market_live_rate 
         ON tbl_watch_list.symbol = tbl_market_live_rate.currency_pair 
         WHERE tbl_watch_list.tbl_user_id = $1 
         ORDER BY tbl_watch_list.tbl_watch_list_id DESC;`,
        [tbl_user_id]
      );
      const marketData = result.rows;
      ws.send(JSON.stringify({ event: 'market_live_rate_data', data: marketData }));
    } catch (err) {
      console.error('Error querying tbl_market_live_rate:', err);
      ws.send(JSON.stringify({ event: 'market_data', error: 'Failed to fetch market data' }));
    }
  };

  sendMarketData();
  const intervalId = setInterval(sendMarketData, 500);

  ws.on('close', () => {
    clearInterval(intervalId);
  });
});

// Handle HTTP upgrade to WebSocket and route to correct namespace
server.on('upgrade', (request, socket, head) => {
  const pathname = url.parse(request.url).pathname;

  if (pathname === '/market_price_update_api') {
    marketPriceNamespace.handleUpgrade(request, socket, head, (ws) => {
      marketPriceNamespace.emit('connection', ws, request);
    });
  } else if (pathname === '/marketlist_api') {
    marketlistNamespace.handleUpgrade(request, socket, head, (ws) => {
      marketlistNamespace.emit('connection', ws, request);
    });
  } else {
    socket.destroy();
  }
});

server.listen(8157, () => {
  console.log('Server is listening on port 8147');
});
